const { PrismaClient } = require('@prisma/client');
const bcrypt = require('bcrypt');

const prisma = new PrismaClient();

const UsuarioController = {
  async createUsuario(req, res) {
    try {
      const { Name, LastName, Phone, Email, Password, Photo, Code, 
        RolId, EnabledId, ConditionId, StateId, UnidadOrganicaId } = req.body;
      // Encrypt the password
      const hashedPassword = await bcrypt.hash(Password, 10);
      const usuario = await prisma.usuario.create(
        {
          data: {
            Name, LastName, Phone, Email, Password: hashedPassword, // Guardamos la contraseña en la base de datos
            Photo, Code, RolId, EnabledId, ConditionId, StateId, UnidadOrganicaId
          }
        }
      );
      res.json(
        {
          status: 'success',
          msj: 'Usuario creado correctamente',
          data: usuario
        }
      );
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al crear la Usuario',
          err: error
        }
      );
    }
  },

  async getAllUsuarios(req, res) {
    try {
      const usuarios = await prisma.usuario.findMany(
        {
          select: {
            id: true, Name: true, LastName: true, Phone: true, 
            Email: true, Password: true, Photo: true, Code: true, 
            RolId: true, EnabledId: true, StateId: true, UnidadOrganicaId: true,
            rol: true, enabled: true, state: true, unidadorganica: true
          },
        }
      );
      if (usuarios.length > 0) {
        res.json(
          {
            status: 'success',
            msj: 'Lista de Usuarios',
            data: usuarios
          }
        );
      } else {
        res.json(
          {
            status: 'warning',
            msj: 'No hay Usuarios registradas'
          }
        );
      }
    } catch (error) {
      console.log(error);
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al obtener la lista de Usuarios',
          err: error
        }
      );
    }
  },

  async getrolUsuario(req, res) {
    try {
      const { elrol } = req.params;
      const usuarios = await prisma.usuario.findMany(
        {
          where: {
            RolId: parseInt(elrol)
          },
          select: {
            id: true, Name: true, LastName: true, Phone: true, 
            Email: true, Password: true, Photo: true, Code: true, 
            RolId: true, EnabledId: true, StateId: true, UnidadOrganicaId: true,
            rol: true, enabled: true, state: true, unidadorganica: true
          },
        }
      );
      if (usuarios.length > 0) {
        res.json(
          {
            status: 'success',
            msj: 'Lista de Usuarios con el rol indicada',
            data: usuarios
          }
        );
      } else {
        res.json(
          {
            status: 'warning',
            msj: 'No hay Usuarios registrados con el rol indicada'
          }
        );
      }
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al obtener la lista de Usuarios',
          err: error
        }
      );
    }
  },

  async getestadoUsuario(req, res) {
    try {
      const { elestado } = req.params;
      const usuarios = await prisma.usuario.findMany(
        {
          where: {
            StateId: parseInt(elestado)
          },
          select: {
            id: true, Name: true, LastName: true, Phone: true, 
            Email: true, Password: true, Photo: true, Code: true, 
            RolId: true, EnabledId: true, StateId: true, UnidadOrganicaId: true,
            rol: true, enabled: true, state: true, unidadorganica: true
          },
        }
      );
      if (usuarios.length > 0) {
        res.json(
          {
            status: 'success',
            msj: 'Lista de Usuarios con el estado indicada',
            data: usuarios
          }
        );
      } else {
        res.json(
          {
            status: 'warning',
            msj: 'No hay Usuarios registradas con el estado indicada'
          }
        );
      }
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al obtener la lista de Usuarios',
          err: error
        }
      );
    }
  },

  async getUsuarioById(req, res) {
    try {
      const { id } = req.params;
      const usuario = await prisma.usuario.findUnique(
        {
          where: {
            id: parseInt(id)
          },
          select: {
            id: true, Name: true, LastName: true, Phone: true, 
            Email: true, Password: true, Photo: true, Code: true, 
            RolId: true, EnabledId: true, StateId: true, UnidadOrganicaId: true,
            rol: true, enabled: true, state: true, unidadorganica: true
          },
        }
      );
      if (usuario !== null) {
        res.json(
          {
            status: 'success',
            msj: 'Usuario encontrada',
            data: usuario
          }
        );
      } else {
        res.json(
          {
            status: 'warning',
            msj: 'No existe la Usuario con el id indicado'
          }
        );
      }
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al obtener la Usuario',
          err: error
        }
      );
    }
  },

  async getUsuarioByRecoverContra(req, res) {
    try {
      const { id, codigo } = req.params;
      const usuario = await prisma.usuario.findUnique(
        {
          where: {
            id: parseInt(id),
            Code: codigo,
            EnabledId: 1
          },
          select: {
            id: true, Name: true, LastName: true, Phone: true, 
            Email: true, Password: true, Photo: true, Code: true, 
            RolId: true, EnabledId: true, StateId: true, UnidadOrganicaId: true,
            rol: true, enabled: true, state: true, unidadorganica: true
          },
        }
      );
      if (usuario !== null) {
        res.json(
          {
            status: 'success',
            msj: 'Usuario encontrada',
            data: usuario
          }
        );
      } else {
        res.json(
          {
            status: 'warning',
            msj: 'No existe la Usuario con el id indicado'
          }
        );
      }
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al obtener el Usuario',
          err: error
        }
      );
    }
  },

  async getUsuarioByPhone(req, res) {
    try {
      const { celular } = req.params;
      const usuario = await prisma.usuario.findUnique(
        {
          where: {
            Phone: celular
          },
          select: {
            id: true, Name: true, LastName: true, Phone: true, 
            Email: true, Password: true, Photo: true, Code: true, 
            RolId: true, EnabledId: true, StateId: true, UnidadOrganicaId: true,
            rol: true, enabled: true, state: true, unidadorganica: true
          },
        }
      );
      if (usuario !== null) {
        res.json(
          {
            status: 'success',
            msj: 'Usuario encontrada',
            data: usuario
          }
        );
      } else {
        res.json(
          {
            status: 'warning',
            msj: 'No existe la Usuario con el id indicado'
          }
        );
      }
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al obtener la Usuario',
          err: error
        }
      );
    }
  },

  async getUsuarioByEmail(req, res) {
    try {
      const { correo } = req.params;
      const usuario = await prisma.usuario.findUnique(
        {
          where: {
            Email: correo
          },
          select: {
            id: true, Name: true, LastName: true, Phone: true, 
            Email: true, Password: true, Photo: true, Code: true, 
            RolId: true, EnabledId: true, StateId: true, UnidadOrganicaId: true,
            rol: true, enabled: true, state: true, unidadorganica: true
          },
        }
      );
      if (usuario !== null) {
        res.json(
          {
            status: 'success',
            msj: 'Usuario encontrada',
            data: usuario
          }
        );
      } else {
        res.json(
          {
            status: 'warning',
            msj: 'No existe la Usuario con el id indicado'
          }
        );
      }
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al obtener la Usuario',
          err: error
        }
      );
    }
  },

  async getUsuarioByLoginEmail(req, res) {
    try {
      const { correo, contra } = req.params;
      const usuario = await prisma.usuario.findUnique(
        {
          where: {
            Email: correo
          },
          select: {
            id: true, Name: true, LastName: true, Phone: true, 
            Email: true, Password: true, Photo: true, Code: true, 
            RolId: true, EnabledId: true, StateId: true, UnidadOrganicaId: true,
            rol: true, enabled: true, state: true, unidadorganica: true
          },
        }
      );
      if (usuario !== null) {
        const parametro = bcrypt.compareSync(contra, usuario.Password);
        if (parametro) {
          res.json(
            {
              status: 'success',
              msj: 'Usuario encontrada',
              data: usuario
            }
          );
        } else {
          res.json(
            {
              status: 'warning',
              msj: 'El Correo es Correcto; pero la Contra es incorrecta'
            }
          );
        }
      } else {
        res.json(
          {
            status: 'warning',
            msj: 'No existe un Usuario con el correo proporcionado'
          }
        );
      }
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al obtener la Usuario',
          err: error
        }
      );
    }
  },

  async getUsuarioByLoginPhone(req, res) {
    try {
      const { celular, contra } = req.params;
      const usuario = await prisma.usuario.findUnique(
        {
          where: {
            Phone: celular
          },
          select: {
            id: true, Name: true, LastName: true, Phone: true, 
            Email: true, Password: true, Photo: true, Code: true, 
            RolId: true, EnabledId: true, StateId: true, UnidadOrganicaId: true,
            rol: true, enabled: true, state: true, unidadorganica: true
          },
        }
      );
      if (usuario !== null) {
        const parametro = bcrypt.compareSync(contra, usuario.Password);
        if (parametro) {
          res.json(
            {
              status: 'success',
              msj: 'Usuario encontrada',
              data: usuario
            }
          );
        } else {
          res.json(
            {
              status: 'warning',
              msj: 'El Correo es Correcto; pero la Contra es incorrecta'
            }
          );
        }
      } else {
        res.json(
          {
            status: 'warning',
            msj: 'No existe un Usuario con el correo proporcionado'
          }
        );
      }
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al obtener la Usuario',
          err: error
        }
      );
    }
  },

  async updateUsuario(req, res) {
    try {
      const { id } = req.params;
      const { Name, LastName, Phone, Email, Password, Photo, Code, 
              RolId, EnabledId, ConditionId, StateId, UnidadOrganicaId } = req.body;
      const usuario = await prisma.usuario.update(
        {
          where: {
            id: parseInt(id)
          },
          data: {
            Name, LastName, Phone, Email, Password, Photo, Code, 
            RolId, EnabledId, ConditionId, StateId, UnidadOrganicaId
          }
        }
      );
      res.json(
        {
          status: 'success',
          msj: 'Usuario actualizada correctamente',
          data: usuario
        }
      );
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al actualizar la Usuario',
          err: error
        }
      );
    }
  },

  async updateUsuarioContra(req, res) {
    try {
      const { id } = req.params;
      const { Name, LastName, Phone, Email, Password, Photo, Code, 
        RolId, EnabledId, ConditionId, StateId, UnidadOrganicaId } = req.body;
      const hashedPassword = await bcrypt.hash(Password, 10);
      const eldato = await prisma.usuario.update(
        {
          where: {
            id: parseInt(id),
            EnabledId: 1
          },
          data: {
            Name, LastName, Phone, Email, Password: hashedPassword, // Guardamos la contraseña en la base de datos
            Photo, Code, RolId, EnabledId, ConditionId, StateId, UnidadOrganicaId
          }
        }
      );
      res.json(
        {
          status: 'success',
          msj: 'Contraseña actualizada correctamente',
          data: eldato
        }
      );
    } catch (error) {
      res.status(500).json(
        {
          status: 'danger',
          msj: 'Error al actualizar la Contraseña',
          err: error
        }
      );
    }
  },

// se oviara la opcion de poder eliminar una Usuario
  // async deleteUsuario(req, res) {
  //   try {
  //     const { id } = req.params;
  //     await prisma.usuario.delete(
  //       {
  //         where: {
  //           id: parseInt(id)
  //         }
  //       }
  //     );
  //     res.json(
  //       {
  //         status: 'warning',
  //         msj: 'Usuario eliminada'
  //       }
  //     );
  //   } catch (error) {
  //     res.status(500).json(
  //       {
  //         status: 'danger',
  //         msj: 'Error al eliminar la Usuario',
  //         err: error
  //       }
  //     );
  //   }
  // }
}

module.exports = UsuarioController;